<?php
session_start();
if (!isset($_SESSION['username']) || $_SESSION['role'] !== 'admin') {
    header("Location: login.html?error=unauthorized");
    exit();
}

require 'db_connect.php';

// Pagination settings
$limit  = 20;
$page   = isset($_GET['page']) ? intval($_GET['page']) : 1;
$page   = max(1, $page);
$offset = ($page - 1) * $limit;

// Count total rows
$total_result   = $conn->query("SELECT COUNT(*) AS total FROM cash_summary");
$total_records  = $total_result->fetch_assoc()['total'];
$total_pages    = ceil($total_records / $limit);

// Fetch paginated rows
$result = $conn->query("
    SELECT *
    FROM cash_summary
    ORDER BY summary_date DESC
    LIMIT $limit OFFSET $offset
");

// Page totals
$page_total_hand  = 0;
$page_total_phone = 0;
$page_total_cash  = 0;

$rows = [];
while ($row = $result->fetch_assoc()) {
    $rows[]            = $row;
    $page_total_hand  += $row['cash_at_hand'];
    $page_total_phone += $row['cash_at_phone'];
    $page_total_cash  += $row['total_cash'];
}

// Grand totals
$grand_res = $conn->query("
    SELECT 
        SUM(cash_at_hand)  AS total_hand,
        SUM(cash_at_phone) AS total_phone,
        SUM(total_cash)    AS total_cash
    FROM cash_summary
");
$grand = $grand_res->fetch_assoc();

// Title for head.php
$page_title = "Cash Summary Records";
?>
<!DOCTYPE html>
<html>
<head>
    <title><?= htmlspecialchars($page_title) ?></title>
    <?php include('head.php'); ?>

    <style>
        body {
            font-family: 'Poppins', Arial, sans-serif;
            background: var(--bg-main);
            margin: 0;
            color: var(--text-main);
        }

        /* Content next to sidebar */
        .content-area {
            margin-left: 260px;
            padding: 25px 30px;
        }

        @media(max-width: 768px) {
            .content-area {
                margin-left: 80px;
                padding: 20px;
            }
        }

        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-end;
            flex-wrap: wrap;
            gap: 10px;
            margin-bottom: 18px;
        }

        .page-title {
            font-size: 22px;
            font-weight: 600;
            color: var(--text-heading);
        }

        .page-subtitle {
            font-size: 13px;
            color: var(--accent-gold);
            margin-top: 4px;
        }

        /* Card around table */
        .page-card {
            background: var(--bg-card);
            border-radius: 14px;
            padding: 18px 22px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.35);
            border: 1px solid #0a4a85;
            color: var(--text-main);
        }

        body[data-theme="light"] .page-card {
            border-color: rgba(148,163,184,0.55);
            box-shadow: 0 12px 28px rgba(15,23,42,0.1);
        }

        .page-card-title {
            font-size: 16px;
            font-weight: 500;
            margin-bottom: 10px;
            color: var(--text-heading);
        }

        /* Scrollable table container */
        .table-container {
            max-height: 500px;
            overflow-y: auto;
            width: 100%;
            margin-top: 6px;
        }

        table {
            width: 100%;
            min-width: 900px;
            border-collapse: collapse;
            background: var(--bg-card);
            box-shadow: 0 0 10px rgba(0,0,0,0.3);
            color: var(--text-main);
        }

        thead tr {
            background: var(--bg-table-head);
        }

        th,
        td {
            padding: 10px 12px;
            border: 1px solid rgba(255,255,255,0.15);
            text-align: center;
            font-size: 13px;
            white-space: nowrap;
        }

        th {
            color: #ffffff;
            font-weight: 600;
            position: sticky;
            top: 0;
            z-index: 2;
        }

        tbody tr {
            background: var(--bg-table-row);
        }

        tbody tr:nth-child(even) {
            background: var(--bg-table-row);
        }

        tbody tr:hover {
            background: var(--bg-table-hover);
            color: #ffffff;
        }

        /* Light theme overrides for table borders/text */
        body[data-theme="light"] th,
        body[data-theme="light"] td {
            border-color: rgba(148,163,184,0.4);
        }

        body[data-theme="light"] th {
            color: #111827;
        }

        body[data-theme="light"] tbody tr {
            background: var(--bg-table-row);
        }

        body[data-theme="light"] tbody tr:nth-child(even) {
            background: var(--bg-table-row);
        }

        body[data-theme="light"] tbody tr:hover {
            background: var(--bg-table-hover);
            color: #111827;
        }

        .numeric {
            text-align: right;
        }

        /* Totals row inside table */
        .totals-row {
            background: #001f3f;
            font-weight: 600;
        }

        .totals-row td {
            border-top: 1px solid rgba(255,255,255,0.25);
        }

        body[data-theme="light"] .totals-row {
            background: #e5e7eb;
        }

        body[data-theme="light"] .totals-row td {
            border-top-color: rgba(148,163,184,0.7);
        }

        /* Grand totals box */
        .grand-totals {
            margin-top: 25px;
            background: var(--bg-card);
            padding: 15px 18px;
            border-left: 6px solid var(--accent-gold);
            border-radius: 10px;
            max-width: 520px;
            margin-left: auto;
            margin-right: auto;
            text-align: center;
            font-weight: 600;
            color: var(--text-main);
            box-shadow: 0 4px 10px rgba(0,0,0,0.3);
        }

        body[data-theme="light"] .grand-totals {
            box-shadow: 0 10px 24px rgba(15,23,42,0.12);
        }

        .grand-totals span.label {
            color: var(--text-main);
        }

        .grand-totals span.value {
            color: var(--accent-gold);
        }

        /* Pagination */
        .pagination {
            text-align: center;
            margin-top: 18px;
        }

        .pagination a {
            padding: 7px 12px;
            margin: 0 4px;
            background: var(--bg-card);
            color: var(--text-main);
            text-decoration: none;
            border-radius: 4px;
            border: 1px solid #0a4a85;
            font-size: 13px;
        }

        .pagination a.active {
            background: #5a47f5;
            border-color: #5a47f5;
            font-weight: bold;
            color: #ffffff;
        }

        .pagination a:hover {
            background: var(--bg-table-hover);
            border-color: var(--bg-table-hover);
            color: #ffffff;
        }

        body[data-theme="light"] .pagination a {
            border-color: rgba(148,163,184,0.6);
        }

        body[data-theme="light"] .pagination a:hover {
            color: #111827;
        }

        footer {
            background: transparent;
        }
    </style>
</head>

<body>

<?php include('sidebar.php'); ?>

<div class="content-area">

    <div class="page-header">
        <div>
            <div class="page-title">Cash Summary Records</div>
            <div class="page-subtitle">
                Submitted by cashiers - page <?= $page ?> of <?= $total_pages ?>
            </div>
        </div>
    </div>

    <div class="page-card">
        <div class="page-card-title">Daily Cash Summary</div>

        <div class="table-container">
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Cash at Hand (KES)</th>
                        <th>Cash at Phone (KES)</th>
                        <th>Total Cash (KES)</th>
                        <th>Cashier</th>
                        <th>Date Submitted</th>
                    </tr>
                </thead>
                <tbody>
                <?php if (count($rows) > 0): ?>
                    <?php foreach ($rows as $r): ?>
                        <tr>
                            <td><?= $r['id'] ?></td>
                            <td class="numeric">KES <?= number_format($r['cash_at_hand'], 2) ?></td>
                            <td class="numeric">KES <?= number_format($r['cash_at_phone'], 2) ?></td>
                            <td class="numeric">KES <?= number_format($r['total_cash'], 2) ?></td>
                            <td><?= htmlspecialchars($r['cashier']) ?></td>
                            <td><?= htmlspecialchars($r['summary_date']) ?></td>
                        </tr>
                    <?php endforeach; ?>

                    <tr class="totals-row">
                        <td>Page Totals</td>
                        <td class="numeric">KES <?= number_format($page_total_hand, 2) ?></td>
                        <td class="numeric">KES <?= number_format($page_total_phone, 2) ?></td>
                        <td class="numeric">KES <?= number_format($page_total_cash, 2) ?></td>
                        <td colspan="2"></td>
                    </tr>
                <?php else: ?>
                    <tr><td colspan="6">No records found.</td></tr>
                <?php endif; ?>
                </tbody>
            </table>
        </div>

        <?php if ($total_pages > 1): ?>
            <div class="pagination">
                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                    <a href="?page=<?= $i ?>" class="<?= ($i == $page ? 'active' : '') ?>"><?= $i ?></a>
                <?php endfor; ?>
            </div>
        <?php endif; ?>
    </div>

    <div class="grand-totals">
        <div>
            <span class="label">Grand Total Cash at Hand:</span>
            <span class="value">KES <?= number_format($grand['total_hand'], 2) ?></span>
        </div>
        <div>
            <span class="label">Grand Total Cash at Phone:</span>
            <span class="value">KES <?= number_format($grand['total_phone'], 2) ?></span>
        </div>
        <div>
            <span class="label">Grand Total Cash:</span>
            <span class="value">KES <?= number_format($grand['total_cash'], 2) ?></span>
        </div>
    </div>

</div>

<?php include 'footer.php'; ?>
</body>
</html>
