<?php
session_start();
if (!isset($_SESSION['username']) || ($_SESSION['role'] ?? '') !== 'cashier') {
    header("Location: login.php");
    exit();
}

require 'db_connect.php';

date_default_timezone_set('Africa/Nairobi');

$page_title = "Submit Cash Summary | Cashier POS";
$active     = 'sales'; // which top-nav item to highlight (Dashboard / Sales / Today’s Sales / Debts)

$success_message = '';
$error_message   = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // Sanitize & cast
    $cash_at_hand  = floatval($_POST['cash_at_hand']  ?? 0);
    $cash_at_phone = floatval($_POST['cash_at_phone'] ?? 0);
    $cashier       = $_SESSION['username']; // use logged-in cashier
    $date          = date('Y-m-d H:i:s');

    // Calculate total cash (server-side, don't trust client total)
    $total_cash = $cash_at_hand + $cash_at_phone;

    // Insert into cash_summary
    $stmt = $conn->prepare("
        INSERT INTO cash_summary (cash_at_hand, cash_at_phone, total_cash, cashier, summary_date)
        VALUES (?, ?, ?, ?, ?)
    ");

    if ($stmt) {
        $stmt->bind_param("dddss", $cash_at_hand, $cash_at_phone, $total_cash, $cashier, $date);

        if ($stmt->execute()) {
            $success_message = "Cash summary submitted successfully. Total Cash: KES " . number_format($total_cash, 2);
        } else {
            $error_message = "Error saving summary: " . htmlspecialchars($stmt->error);
        }

        $stmt->close();
    } else {
        $error_message = "Error preparing statement: " . htmlspecialchars($conn->error);
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <?php include 'cashier_head.php'; ?>
    <style>
        .cashier-main {
            padding: 24px 30px 34px;
        }

        @media (max-width: 900px) {
            .cashier-main {
                padding: 18px 16px 26px;
            }
        }

        .cash-summary-wrapper {
            max-width: 640px;
            margin: 0 auto;
        }

        .pos-card {
            background:
                radial-gradient(circle at top left, rgba(34,211,238,0.14), transparent 55%),
                linear-gradient(135deg, #064ab8 0%, #022a6b 55%, #021d53 100%);
            border-radius: 18px;
            padding: 18px 22px 22px;
            box-shadow: 0 20px 45px rgba(0,0,0,0.55);
            color: #E9F5FF;
            border: 1px solid rgba(130,238,253,0.5);
        }

        html[data-theme="light"] .pos-card {
            background:
                radial-gradient(circle at top left, rgba(56,189,248,0.16), transparent 55%),
                #ffffff;
            box-shadow: 0 12px 26px rgba(15,23,42,0.14);
            border-color: #dbeafe;
            color: #0f172a;
        }

        .pos-card-header h2 {
            font-size: 19px;
            font-weight: 600;
        }

        .pos-card-header p {
            font-size: 13px;
            color: #c3d8ff;
            margin-top: 4px;
        }

        html[data-theme="light"] .pos-card-header p {
            color: #64748b;
        }

        .cash-summary-form {
            margin-top: 14px;
        }

        .cash-summary-form label {
            display: block;
            font-size: 13px;
            font-weight: 500;
            margin-bottom: 4px;
            color: #E9F5FF;
        }

        html[data-theme="light"] .cash-summary-form label {
            color: #0f172a;
        }

        .cash-summary-form input[type="number"],
        .cash-summary-form input[type="text"] {
            width: 100%;
            padding: 9px 11px;
            border-radius: 10px;
            border: 1px solid rgba(130,238,253,0.5);
            background: #0a3d87;
            color: #E9F5FF;
            font-size: 13px;
            margin-bottom: 10px;
        }

        .cash-summary-form input::placeholder {
            color: rgba(226,232,240,0.7);
        }

        html[data-theme="light"] .cash-summary-form input[type="number"],
        html[data-theme="light"] .cash-summary-form input[type="text"] {
            background: #ffffff;
            color: #0f172a;
            border-color: #cbd5f5;
        }

        html[data-theme="light"] .cash-summary-form input::placeholder {
            color: #9ca3af;
        }

        .cash-summary-form input:focus {
            outline: none;
            border-color: #00FFE5;
            box-shadow: 0 0 0 1px rgba(0,255,229,0.35);
        }

        html[data-theme="light"] .cash-summary-form input:focus {
            border-color: #0ea5e9;
            box-shadow: 0 0 0 1px rgba(56,189,248,0.35);
        }

        .cash-summary-form input[readonly] {
            background: rgba(15,23,42,0.7);
            cursor: default;
        }

        html[data-theme="light"] .cash-summary-form input[readonly] {
            background: #f3f4f6;
        }

        .btn-primary {
            padding: 10px 18px;
            border-radius: 999px;
            border: none;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            background: linear-gradient(to right, #22d3ee, #0ea5e9);
            color: #011b35;
            width: 100%;
            margin-top: 4px;
            box-shadow: 0 10px 24px rgba(0,0,0,0.45);
        }

        .btn-primary:hover {
            filter: brightness(1.05);
        }

        .form-footer-links {
            margin-top: 12px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 10px;
            flex-wrap: wrap;
        }

        .back-link {
            text-decoration: none;
            font-size: 13px;
            color: #82EEFD;
        }

        .back-link:hover {
            text-decoration: underline;
        }

        html[data-theme="light"] .back-link {
            color: #0369a1;
        }

        .pill-info {
            font-size: 11px;
            padding: 4px 9px;
            border-radius: 999px;
            background: rgba(15,23,42,0.86);
            color: #c3d8ff;
        }

        html[data-theme="light"] .pill-info {
            background: #eff6ff;
            color: #0f172a;
        }

        .alert {
            margin-bottom: 10px;
            padding: 8px 10px;
            border-radius: 10px;
            font-size: 13px;
        }

        .alert-success {
            background: rgba(34,197,94,0.12);
            border: 1px solid rgba(34,197,94,0.7);
            color: #bbf7d0;
        }

        html[data-theme="light"] .alert-success {
            background: #ecfdf3;
            border-color: #22c55e;
            color: #166534;
        }

        .alert-error {
            background: rgba(239,68,68,0.14);
            border: 1px solid rgba(248,113,113,0.8);
            color: #fecaca;
        }

        html[data-theme="light"] .alert-error {
            background: #fef2f2;
            border-color: #f97373;
            color: #b91c1c;
        }
    </style>
    <script>
        function updateTotal() {
            const hand  = parseFloat(document.getElementById('cash_at_hand').value)  || 0;
            const phone = parseFloat(document.getElementById('cash_at_phone').value) || 0;
            document.getElementById('total_cash').value = (hand + phone).toFixed(2);
        }
    </script>
</head>
<body>

<div class="cashier-shell">
    <?php include "cashier_sidebar.php"; ?>

    <main class="cashier-main">

        <!-- Consistent top bar with nav + theme toggle -->
        <div class="cashier-topbar">
            <div class="cashier-topbar-left">
                <h1>Submit cash summary</h1>
                <small>Record the cash you are handing over at the end of your shift.</small>
            </div>

            <div class="cashier-topbar-right">
                <nav class="cashier-topnav">
                    <a href="cashier_dashboard.php" class="<?= ($active === 'dashboard') ? 'active' : '' ?>">Dashboard</a>
                    <a href="record_sales.php" class="<?= ($active === 'sales') ? 'active' : '' ?>">Sales</a>
                    <a href="cashier_view_daily_sales.php" class="<?= ($active === 'daily_sales') ? 'active' : '' ?>">Today’s Sales</a>
                    <a href="customer_debts.php" class="<?= ($active === 'debts') ? 'active' : '' ?>">Debts</a>
                </nav>

                <button type="button" class="cashier-theme-toggle" onclick="toggleCashierTheme()">
                    <span class="icon">🌓</span>
                    <span class="label">Theme</span>
                </button>

                <button class="cashier-toggle" type="button" onclick="toggleCashierSidebar()">
                    ☰
                </button>
            </div>
        </div>

        <div class="cash-summary-wrapper">
            <section class="pos-card">
                <div class="pos-card-header">
                    <h2>End-of-shift cash summary</h2>
                    <p>Enter cash in the drawer and in mobile money, the system will compute the total.</p>
                </div>

                <?php if ($success_message): ?>
                    <div class="alert alert-success">
                        <?= htmlspecialchars($success_message) ?>
                    </div>
                <?php endif; ?>

                <?php if ($error_message): ?>
                    <div class="alert alert-error">
                        <?= htmlspecialchars($error_message) ?>
                    </div>
                <?php endif; ?>

                <form class="cash-summary-form" action="submit_cash_summary.php" method="post">
                    <label for="cash_at_hand">Cash at hand (physical cash)</label>
                    <input
                        type="number"
                        step="0.01"
                        id="cash_at_hand"
                        name="cash_at_hand"
                        oninput="updateTotal()"
                        placeholder="Enter cash at hand"
                        required
                    >

                    <label for="cash_at_phone">Cash at phone (Paybill / Till)</label>
                    <input
                        type="number"
                        step="0.01"
                        id="cash_at_phone"
                        name="cash_at_phone"
                        oninput="updateTotal()"
                        placeholder="Enter M-Pesa / mobile collections"
                        required
                    >

                    <label for="total_cash">Total cash</label>
                    <input
                        type="number"
                        step="0.01"
                        id="total_cash"
                        name="total_cash"
                        readonly
                    >

                    <label for="cashier">Cashier</label>
                    <input
                        type="text"
                        id="cashier"
                        value="<?= htmlspecialchars($_SESSION['username'] ?? '') ?>"
                        readonly
                    >

                    <button type="submit" class="btn-primary">Submit summary</button>

                    <div class="form-footer-links">
                        <a href="cashier_dashboard.php" class="back-link">← Back to dashboard</a>
                        <span class="pill-info">Summary time: <?= date('Y-m-d H:i') ?></span>
                    </div>
                </form>
            </section>
        </div>

    </main>
</div>

<?php include 'footer.php'; ?>
</body>
</html>
