<?php
session_start();
if (!isset($_SESSION['username']) || ($_SESSION['role'] ?? '') !== 'cashier') {
    header("Location: login.php");
    exit();
}

include("db_connect.php");

$error = "";

// Validate & fetch ID
if (!isset($_GET['id']) || !ctype_digit($_GET['id'])) {
    die("Invalid customer ID.");
}
$id = (int)$_GET['id'];

// Fetch customer using prepared statement
$stmt = $conn->prepare("SELECT id, customer_name, phone, created_at FROM customers WHERE id = ?");
$stmt->bind_param("i", $id);
$stmt->execute();
$result   = $stmt->get_result();
$customer = $result->fetch_assoc();
$stmt->close();

if (!$customer) {
    die("Customer not found.");
}

// Handle update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name  = trim($_POST['customer_name'] ?? '');
    $phone = trim($_POST['phone'] ?? '');

    if ($name === '' || $phone === '') {
        $error = "Please fill in all fields.";
    } else {
        $update = $conn->prepare("UPDATE customers SET customer_name = ?, phone = ? WHERE id = ?");
        $update->bind_param("ssi", $name, $phone, $id);

        if ($update->execute()) {
            $update->close();
            header("Location: cashier_customer_details.php?updated=1");
            exit();
        } else {
            $error = "Error updating customer: " . htmlspecialchars($update->error);
            $update->close();
        }
    }

    // Update local variables for sticky form
    $customer['customer_name'] = $name;
    $customer['phone']         = $phone;
}

$page_title = "Edit Customer | Cashier POS";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <?php include 'cashier_head.php'; ?>
    <style>
        /* ========= Layout + sticky footer ========= */
        html, body {
            height: 100%;
        }
        body {
            min-height: 100vh;
            margin: 0;
            display: flex;
            flex-direction: column;
        }
        .cashier-shell {
            flex: 1;
            display: flex;
            min-height: 0;
        }
        .cashier-main {
            flex: 1;
            min-width: 0;
            padding: 24px 30px 34px;
        }
        footer {
            position: relative !important;
            bottom: auto !important;
            width: 100%;
            margin-top: auto;
        }

        /* ========= HERO TOP BAR ========= */
        .cashier-hero-bar {
            background: linear-gradient(90deg, #e0f7ff 0%, #f7fbff 35%, #e6f5ff 100%);
            border-radius: 14px;
            padding: 14px 22px;
            margin-bottom: 18px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 20px;
            border: 1px solid rgba(148,163,184,0.35);
        }
        .hero-left h1 {
            font-size: 20px;
            font-weight: 600;
            margin: 0;
            color: #020617;
        }
        .hero-left p {
            margin: 2px 0 0;
            font-size: 13px;
            color: #64748b;
        }
        .hero-nav {
            display: flex;
            align-items: center;
            gap: 16px;
            flex-wrap: wrap;
        }
        .hero-tabs {
            display: flex;
            gap: 12px;
            flex-wrap: wrap;
        }
        .hero-tab {
            font-size: 13px;
            padding: 7px 14px;
            border-radius: 999px;
            text-decoration: none;
            color: #0f172a;
            background: transparent;
        }
        .hero-tab:hover {
            background: rgba(148,163,184,0.18);
        }
        .hero-tab.active {
            background: #06b6d4;
            color: #f9fafb;
            font-weight: 600;
        }
        .hero-actions {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .hero-menu-btn {
            width: 36px;
            height: 36px;
            border-radius: 999px;
            border: none;
            background: #06b6d4;
            color: #f9fafb;
            font-size: 18px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
        }
        .hero-menu-btn:focus {
            outline: none;
        }

        /* ========= Theme toggle pill ========= */
        .theme-toggle-pill {
            display: inline-flex;
            align-items: center;
            gap: 6px;
            border-radius: 999px;
            padding: 4px 10px;
            background: #0f172a;
            color: #e5e7eb;
            font-size: 12px;
            cursor: pointer;
            border: 1px solid rgba(148,163,184,0.7);
        }
        .theme-toggle-pill-circle {
            width: 16px;
            height: 16px;
            border-radius: 999px;
            background: linear-gradient(135deg, #facc15, #fb923c);
        }

        /* ========= Card / form ========= */
        .pos-card {
            background: linear-gradient(135deg, #064ab8 0%, #022a6b 55%, #021d53 100%);
            border-radius: 18px;
            padding: 18px 22px 22px;
            box-shadow: 0 20px 45px rgba(0,0,0,0.55);
            color: #E9F5FF;
            border: 1px solid rgba(130,238,253,0.5);
            max-width: 520px;
            margin: 0 auto;
        }

        .pos-card-header h2 {
            font-size: 19px;
            font-weight: 600;
        }

        .pos-card-header p {
            font-size: 13px;
            color: #c3d8ff;
            margin-top: 4px;
        }

        .customer-form {
            margin-top: 14px;
        }

        .customer-form label {
            display: block;
            font-size: 13px;
            font-weight: 500;
            margin-bottom: 4px;
            color: #E9F5FF;
        }

        .customer-form input[type="text"] {
            width: 100%;
            padding: 9px 11px;
            border-radius: 10px;
            border: 1px solid rgba(130,238,253,0.5);
            background: #0a3d87;
            color: #E9F5FF;
            font-size: 13px;
            margin-bottom: 10px;
        }

        .customer-form input::placeholder {
            color: rgba(226,232,240,0.7);
        }

        .customer-form input:focus {
            outline: none;
            border-color: #00FFE5;
            box-shadow: 0 0 0 1px rgba(0,255,229,0.35);
        }

        .btn-primary {
            padding: 10px 18px;
            border-radius: 999px;
            border: none;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            background: var(--cash-accent-turquoise);
            color: #011b35;
            width: 100%;
            margin-top: 4px;
        }

        .btn-primary:hover {
            filter: brightness(1.08);
        }

        .alert-error {
            margin-top: 10px;
            margin-bottom: 8px;
            padding: 8px 10px;
            border-radius: 10px;
            font-size: 13px;
            background: rgba(239,68,68,0.14);
            border: 1px solid rgba(248,113,113,0.8);
            color: #fecaca;
        }

        .form-footer-links {
            margin-top: 12px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 10px;
            flex-wrap: wrap;
        }

        .back-link {
            text-decoration: none;
            font-size: 13px;
            color: #82EEFD;
        }

        .back-link:hover {
            text-decoration: underline;
        }

        .pill-info {
            font-size: 11px;
            padding: 4px 9px;
            border-radius: 999px;
            background: rgba(15,23,42,0.86);
            color: #c3d8ff;
        }

        @media (max-width: 960px) {
            .cashier-main {
                padding: 18px 16px 26px;
            }
            .cashier-hero-bar {
                flex-direction: column;
                align-items: flex-start;
            }
            .hero-nav {
                width: 100%;
                justify-content: space-between;
            }
        }

        /* ========= LIGHT THEME OVERRIDES ========= */

        html[data-theme="light"] .cashier-hero-bar,
        body[data-theme="light"] .cashier-hero-bar,
        body.light-theme .cashier-hero-bar {
            background: linear-gradient(90deg, #f1fbff 0%, #ffffff 35%, #f5fbff 100%);
            border-color: rgba(148,163,184,0.4);
        }

        html[data-theme="light"] .hero-left h1,
        body[data-theme="light"] .hero-left h1,
        body.light-theme .hero-left h1 {
            color: #020617;
        }

        html[data-theme="light"] .hero-left p,
        body[data-theme="light"] .hero-left p,
        body.light-theme .hero-left p {
            color: #64748b;
        }

        html[data-theme="light"] .pos-card,
        body[data-theme="light"] .pos-card,
        body.light-theme .pos-card {
            background: #ffffff !important;
            color: #0f172a !important;
            border-color: rgba(148,163,184,0.6) !important;
            box-shadow: 0 8px 22px rgba(15,23,42,0.12);
        }

        html[data-theme="light"] .pos-card-header p,
        body[data-theme="light"] .pos-card-header p,
        body.light-theme .pos-card-header p {
            color: #64748b !important;
        }

        html[data-theme="light"] .customer-form label,
        body[data-theme="light"] .customer-form label,
        body.light-theme .customer-form label {
            color: #0f172a !important;
        }

        html[data-theme="light"] .customer-form input[type="text"],
        body[data-theme="light"] .customer-form input[type="text"],
        body.light-theme .customer-form input[type="text"] {
            background: #e5f2ff !important;
            color: #0f172a !important;
            border-color: rgba(148,163,184,0.9) !important;
        }

        html[data-theme="light"] .back-link,
        body[data-theme="light"] .back-link,
        body.light-theme .back-link {
            color: #0369a1;
        }

        html[data-theme="light"] .pill-info,
        body[data-theme="light"] .pill-info,
        body.light-theme .pill-info {
            background: #e2e8f0;
            color: #0f172a;
        }

        html[data-theme="light"] .theme-toggle-pill,
        body[data-theme="light"] .theme-toggle-pill,
        body.light-theme .theme-toggle-pill {
            background: #e2e8f0;
            color: #0f172a;
            border-color: rgba(148,163,184,0.9);
        }
    </style>
</head>
<body>

<div class="cashier-shell">
    <?php include "cashier_sidebar.php"; ?>

    <main class="cashier-main">

        <!-- HERO TOP BAR with tabs + theme toggle -->
        <header class="cashier-hero-bar">
            <div class="hero-left">
                <h1>Edit Customer | Cashier POS</h1>
                <p>Update name or phone number for this customer record.</p>
            </div>

            <div class="hero-nav">
                <nav class="hero-tabs">
                    <a href="cashier_dashboard.php" class="hero-tab">Dashboard</a>
                    <a href="record_sale.php" class="hero-tab">Sales</a>
                    <a href="view_today_sales.php" class="hero-tab">Today’s Sales</a>
                    <a href="cashier_transactions.php" class="hero-tab">Transactions</a>
                    <a href="cashier_customer_details.php" class="hero-tab active">Customers</a>
                    <a href="add_expense.php" class="hero-tab">Expenses</a>
                    <a href="customer_debts.php" class="hero-tab">Debts</a>
                    <a href="add_damages.php" class="hero-tab">Damaged</a>
                </nav>

                <div class="hero-actions">
                    <button type="button" class="theme-toggle-pill" id="themeToggle">
                        <span class="theme-toggle-pill-circle"></span>
                        <span id="themeToggleLabel">Theme</span>
                    </button>
                    <button class="hero-menu-btn" type="button" onclick="toggleCashierSidebar()">
                        ☰
                    </button>
                </div>
            </div>
        </header>

        <!-- Edit customer card -->
        <section class="pos-card">
            <div class="pos-card-header">
                <h2>Edit customer</h2>
                <p>Customer ID: <?= (int)$customer['id'] ?></p>
            </div>

            <?php if ($error): ?>
                <div class="alert-error">
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <form method="POST" class="customer-form">
                <label for="customer_name">Customer name</label>
                <input
                    type="text"
                    id="customer_name"
                    name="customer_name"
                    value="<?= htmlspecialchars($customer['customer_name']) ?>"
                    required
                >

                <label for="phone">Phone number</label>
                <input
                    type="text"
                    id="phone"
                    name="phone"
                    value="<?= htmlspecialchars($customer['phone']) ?>"
                    required
                >

                <button type="submit" class="btn-primary">Update customer</button>

                <div class="form-footer-links">
                    <a href="cashier_customer_details.php" class="back-link">← Back to customers</a>
                    <?php if (!empty($customer['created_at'])): ?>
                        <span class="pill-info">
                            Created: <?= htmlspecialchars(date('Y-m-d H:i', strtotime($customer['created_at']))) ?>
                        </span>
                    <?php endif; ?>
                </div>
            </form>
        </section>

    </main>
</div>

<?php include 'footer.php'; ?>

<script>
    // Simple theme toggler: toggles html[data-theme] between "dark" and "light"
    (function () {
        const root = document.documentElement;
        const body = document.body;
        const toggleBtn = document.getElementById('themeToggle');
        const label = document.getElementById('themeToggleLabel');

        function applyTheme(theme) {
            root.setAttribute('data-theme', theme);
            body.setAttribute('data-theme', theme);
            body.classList.toggle('light-theme', theme === 'light');
            localStorage.setItem('cashierTheme', theme);
            if (label) {
                label.textContent = theme === 'light' ? 'Light' : 'Dark';
            }
        }

        function initTheme() {
            const saved = localStorage.getItem('cashierTheme');
            const initial = saved === 'light' || saved === 'dark' ? saved : 'dark';
            applyTheme(initial);
        }

        if (toggleBtn) {
            toggleBtn.addEventListener('click', function () {
                const current = root.getAttribute('data-theme') === 'light' ? 'light' : 'dark';
                const next = current === 'light' ? 'dark' : 'light';
                applyTheme(next);
            });
        }

        document.addEventListener('DOMContentLoaded', initTheme);
    })();
</script>

</body>
</html>
