<?php
session_start();
if (!isset($_SESSION['username']) || ($_SESSION['role'] ?? '') !== 'admin') {
    header("Location: login.php");
    exit();
}

include("db_connect.php");

$error = "";

// Validate & fetch ID
if (!isset($_GET['id']) || !ctype_digit($_GET['id'])) {
    die("Invalid customer ID.");
}
$id = (int)$_GET['id'];

// Fetch customer using prepared statement
$stmt = $conn->prepare("SELECT id, customer_name, phone, created_at FROM customers WHERE id = ?");
$stmt->bind_param("i", $id);
$stmt->execute();
$result   = $stmt->get_result();
$customer = $result->fetch_assoc();
$stmt->close();

if (!$customer) {
    die("Customer not found.");
}

// Handle update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name  = trim($_POST['customer_name'] ?? '');
    $phone = trim($_POST['phone'] ?? '');

    if ($name === '' || $phone === '') {
        $error = "Please fill in all fields.";
    } else {
        $update = $conn->prepare("UPDATE customers SET customer_name = ?, phone = ? WHERE id = ?");
        $update->bind_param("ssi", $name, $phone, $id);

        if ($update->execute()) {
            $update->close();
            header("Location: cashier_customer_details.php?updated=1");
            exit();
        } else {
            $error = "Error updating customer: " . htmlspecialchars($update->error);
            $update->close();
        }
    }

    // Update local variables for sticky form
    $customer['customer_name'] = $name;
    $customer['phone']         = $phone;
}

$page_title = "Edit Customer | Cashier POS";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <?php include 'cashier_head.php'; ?>
    <style>
        .cashier-main {
            padding: 24px 30px 34px;
        }

        .pos-card {
            background: linear-gradient(135deg, #064ab8 0%, #022a6b 55%, #021d53 100%);
            border-radius: 18px;
            padding: 18px 22px 22px;
            box-shadow: 0 20px 45px rgba(0,0,0,0.55);
            color: #E9F5FF;
            border: 1px solid rgba(130,238,253,0.5);
            max-width: 520px;
            margin: 0 auto;
        }

        .pos-card-header h2 {
            font-size: 19px;
            font-weight: 600;
        }

        .pos-card-header p {
            font-size: 13px;
            color: #c3d8ff;
            margin-top: 4px;
        }

        .customer-form {
            margin-top: 14px;
        }

        .customer-form label {
            display: block;
            font-size: 13px;
            font-weight: 500;
            margin-bottom: 4px;
            color: #E9F5FF;
        }

        .customer-form input[type="text"] {
            width: 100%;
            padding: 9px 11px;
            border-radius: 10px;
            border: 1px solid rgba(130,238,253,0.5);
            background: #0a3d87;
            color: #E9F5FF;
            font-size: 13px;
            margin-bottom: 10px;
        }

        .customer-form input::placeholder {
            color: rgba(226,232,240,0.7);
        }

        .customer-form input:focus {
            outline: none;
            border-color: #00FFE5;
            box-shadow: 0 0 0 1px rgba(0,255,229,0.35);
        }

        .btn-primary {
            padding: 10px 18px;
            border-radius: 999px;
            border: none;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            background: var(--cash-accent-turquoise);
            color: #011b35;
            width: 100%;
            margin-top: 4px;
        }

        .btn-primary:hover {
            filter: brightness(1.08);
        }

        .alert-error {
            margin-top: 10px;
            margin-bottom: 8px;
            padding: 8px 10px;
            border-radius: 10px;
            font-size: 13px;
            background: rgba(239,68,68,0.14);
            border: 1px solid rgba(248,113,113,0.8);
            color: #fecaca;
        }

        .form-footer-links {
            margin-top: 12px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 10px;
            flex-wrap: wrap;
        }

        .back-link {
            text-decoration: none;
            font-size: 13px;
            color: #82EEFD;
        }

        .back-link:hover {
            text-decoration: underline;
        }

        .pill-info {
            font-size: 11px;
            padding: 4px 9px;
            border-radius: 999px;
            background: rgba(15,23,42,0.86);
            color: #c3d8ff;
        }

        @media (max-width: 900px) {
            .cashier-main {
                padding: 18px 16px 26px;
            }
        }
    </style>
</head>
<body>

<div class="cashier-shell">
    <?php include "cashier_sidebar.php"; ?>

    <main class="cashier-main">

        <div class="cashier-topbar">
            <div>
                <h1>Edit customer</h1>
                <small>Update name or phone number for this customer record.</small>
            </div>
            <button class="cashier-toggle" onclick="toggleCashierSidebar()">Menu</button>
        </div>

        <section class="pos-card">
            <div class="pos-card-header">
                <h2>Edit customer</h2>
                <p>Customer ID: <?= (int)$customer['id'] ?></p>
            </div>

            <?php if ($error): ?>
                <div class="alert-error">
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <form method="POST" class="customer-form">
                <label for="customer_name">Customer name</label>
                <input
                    type="text"
                    id="customer_name"
                    name="customer_name"
                    value="<?= htmlspecialchars($customer['customer_name']) ?>"
                    required
                >

                <label for="phone">Phone number</label>
                <input
                    type="text"
                    id="phone"
                    name="phone"
                    value="<?= htmlspecialchars($customer['phone']) ?>"
                    required
                >

                <button type="submit" class="btn-primary">Update customer</button>

                <div class="form-footer-links">
                    <a href="cashier_customer_details.php" class="back-link">← Back to customers</a>
                    <?php if (!empty($customer['created_at'])): ?>
                        <span class="pill-info">
                            Created: <?= htmlspecialchars(date('Y-m-d H:i', strtotime($customer['created_at']))) ?>
                        </span>
                    <?php endif; ?>
                </div>
            </form>
        </section>

    </main>
</div>

<?php include 'footer.php'; ?>
</body>
</html>
